{-
  Copyright © 2023 Josep Bigorra

  This file is part of Keuringsdienst.
  Keuringsdienst is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation,
    either version 3 of the License, or (at your option) any later version.

  Keuringsdienst is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  See the GNU General Public License for more details.
  You should have received a copy of the GNU General Public License along with Keuringsdienst.
  If not, see <https://www.gnu.org/licenses/>.
-}
{-# OPTIONS_GHC -F -pgmF htfpp #-}

module KeuringsdienstSpec
  ( htf_thisModulesTests,
  )
where

import Data.Text
import Keuringsdienst as K
import Test.Framework

-- λ test helpers
someErrorMessage :: Text
someErrorMessage = "You're simply a test! Better than all the rest!"

alwaysSuccessfulValidationRule :: ValidationRule a
alwaysSuccessfulValidationRule = ValidationRule {performValidation = const K.Success}

alwaysFailureValidationRule :: ValidationRule a
alwaysFailureValidationRule = ValidationRule {performValidation = const (K.Failure [someErrorMessage])}

-- λ test cases
test_validationResultsCanBeComposedIfBothSuccess :: IO ()
test_validationResultsCanBeComposedIfBothSuccess =
  assertEqual K.Success systemUnderTest
  where
    systemUnderTest :: ValidationResult
    systemUnderTest = K.Success <> K.Success

test_validationResultsCanBeComposedIfSuccessFailure :: IO ()
test_validationResultsCanBeComposedIfSuccessFailure =
  assertEqual expected systemUnderTest
  where
    expected :: ValidationResult
    expected = K.Failure [someErrorMessage]
    systemUnderTest :: ValidationResult
    systemUnderTest = K.Success <> K.Failure [someErrorMessage]

test_validationResultsCanBeComposedIfFailureSuccess :: IO ()
test_validationResultsCanBeComposedIfFailureSuccess =
  assertEqual expected systemUnderTest
  where
    expected :: ValidationResult
    expected = K.Failure [someErrorMessage]
    systemUnderTest :: ValidationResult
    systemUnderTest = K.Failure [someErrorMessage] <> K.Success

test_keurenOperatorAppliesValidation :: IO ()
test_keurenOperatorAppliesValidation =
  assertEqual K.Success systemUnderTest
  where
    systemUnderTest :: ValidationResult
    systemUnderTest = ("" :: Text) |?| alwaysSuccessfulValidationRule

test_nonInfixKeurenOperatorAppliesValidation :: IO ()
test_nonInfixKeurenOperatorAppliesValidation =
  assertEqual K.Success systemUnderTest
  where
    systemUnderTest :: ValidationResult
    systemUnderTest = keuren ("" :: Text) alwaysSuccessfulValidationRule

test_nonInfixValidateOperatorAppliesValidation :: IO ()
test_nonInfixValidateOperatorAppliesValidation =
  assertEqual K.Success systemUnderTest
  where
    systemUnderTest :: ValidationResult
    systemUnderTest = validate ("" :: Text) alwaysSuccessfulValidationRule

test_misschienKeurenOperatorAppliesValidation :: IO ()
test_misschienKeurenOperatorAppliesValidation =
  assertEqual expected systemUnderTest
  where
    expected :: ValidationResult
    expected = K.Failure [someErrorMessage]
    systemUnderTest :: ValidationResult
    systemUnderTest = (Just "" :: Maybe Text) |??| alwaysFailureValidationRule

test_nonInfixMisschienKeurenOperatorAppliesValidation :: IO ()
test_nonInfixMisschienKeurenOperatorAppliesValidation =
  assertEqual expected systemUnderTest
  where
    expected :: ValidationResult
    expected = K.Failure [someErrorMessage]
    systemUnderTest :: ValidationResult
    systemUnderTest = misschienKeuren (Just "" :: Maybe Text) alwaysFailureValidationRule

test_nonInfixMaybeValidateOperatorAppliesValidation :: IO ()
test_nonInfixMaybeValidateOperatorAppliesValidation =
  assertEqual expected systemUnderTest
  where
    expected :: ValidationResult
    expected = K.Failure [someErrorMessage]
    systemUnderTest :: ValidationResult
    systemUnderTest = maybeValidate (Just "" :: Maybe Text) alwaysFailureValidationRule

test_misschienKeurenOperatorAppliesValidationOnNothing :: IO ()
test_misschienKeurenOperatorAppliesValidationOnNothing =
  assertEqual K.Success systemUnderTest
  where
    systemUnderTest :: ValidationResult
    systemUnderTest = (Nothing :: Maybe Text) |??| alwaysSuccessfulValidationRule

test_nonInfixMisschienKeurenOperatorAppliesValidationOnNothing :: IO ()
test_nonInfixMisschienKeurenOperatorAppliesValidationOnNothing =
  assertEqual K.Success systemUnderTest
  where
    systemUnderTest :: ValidationResult
    systemUnderTest = misschienKeuren (Nothing :: Maybe Text) alwaysSuccessfulValidationRule

test_nonInfixMaybeValidateOperatorAppliesValidationOnNothing :: IO ()
test_nonInfixMaybeValidateOperatorAppliesValidationOnNothing =
  assertEqual K.Success systemUnderTest
  where
    systemUnderTest :: ValidationResult
    systemUnderTest = maybeValidate (Nothing :: Maybe Text) alwaysSuccessfulValidationRule
