{- | Tools for building Gopher menus.

Simple, intuitive functions for creating Gopher menus

Intentionally not using rigid definition of types for the menu,
I have just found that annoying in the past.

-}
module Venusia.MenuBuilder 
  ( -- * Menu creation functions
    menu
  , text
  , directory
  , search
  , binary
  , image
  , gif
  , html
  , info
  , error
  -- * Customization
  , item
  -- * Rendering
  , render
  ) where

import Venusia.Types
import qualified Data.ByteString.Char8 as BS
import Prelude hiding (error)

type ItemType = Char

type ItemHelper = BS.ByteString -> Selector -> BS.ByteString -> Int -> BS.ByteString

crlf :: BS.ByteString
crlf = BS.pack "\r\n"

terminator :: BS.ByteString
terminator = BS.pack ".\r\n"

-- | Create a complete Gopher menu from menu items and render it
menu :: [BS.ByteString] -> BS.ByteString
menu items = BS.concat items `BS.append` terminator

-- | Create a standard text file item
text :: ItemHelper
text = item '0'

-- | Create a directory menu item
directory :: ItemHelper
directory = item '1'

-- | Create a search item
search :: ItemHelper
search = item '7'

-- | Create a binary file item
binary :: ItemHelper
binary = item '9'

-- | Create an image file item
image :: ItemHelper
image = item 'I'

-- | Create a GIF image item
gif :: ItemHelper
gif = item 'g'

-- | Create an HTML document item
html :: ItemHelper
html display url = item 'h' display ("URL:" <> url)

-- | Create an error item
error :: BS.ByteString -> BS.ByteString
error msg = item '3' msg BS.empty BS.empty 0

-- | Create an information line (no selector)
info :: BS.ByteString -> BS.ByteString
info msg = item 'i' msg BS.empty BS.empty 0

-- | Create a custom item with any type character
item :: ItemType -> BS.ByteString -> Selector -> BS.ByteString -> Int -> BS.ByteString
item typeChar display selector host port =
  BS.singleton typeChar `BS.append`
  display `BS.append`
  BS.singleton '\t' `BS.append`
  selector `BS.append` 
  BS.singleton '\t' `BS.append`
  host `BS.append`
  BS.singleton '\t' `BS.append`
  BS.pack (show port) `BS.append`
  crlf

-- | Alias for menu function
render :: [BS.ByteString] -> BS.ByteString
render = menu